package gov.va.genisis2.dao.impl;

import gov.va.genisis2.dao.IStudyApprovalDao;
import gov.va.genisis2.model.StudyApproval;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Root;

import org.hibernate.query.Query;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.hibernate5.HibernateTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class StudyApprovalDao.
 *
 * The StudyApproval data access object (DAO) is an object that provides an abstract
 * interface to some type of database or other persistence mechanism. 
 * By mapping application calls to the persistence layer, StudyApproval
 * DAO provide some specific data operations without exposing details of the database.
 */
@Repository
@Transactional(value = "transactionManager")
public class StudyApprovalDao implements IStudyApprovalDao {
	
	/** The logger. */
	private final org.slf4j.Logger logger = LoggerFactory.getLogger(StudyApprovalDao.class);

	/** The hibernate template. */
	private HibernateTemplate hibernateTemplate;

	/**
	 * Gets the hibernate template.
	 *
	 * @return HibernateTemplate This returns HibernateTemplate.
	 */
	public HibernateTemplate getHibernateTemplate() {
		return hibernateTemplate;
	}

	/**
	 * Sets the hibernate template.
	 *
	 * @param hibernateTemplate The hibernateTemplate.
	 */
	@Autowired
	public void setHibernateTemplate(HibernateTemplate hibernateTemplate) {
		this.hibernateTemplate = hibernateTemplate;
	}

	/**
	 * This method is used to createStudyApproval.
	 * @param entity The entity.
	 * @return int This returns entity id.
	 */
	@Override
	public int createStudyApproval(StudyApproval entity) {
		int id = 0;
		
		if (entity==null)
		{
			return 0;
		}
		entity.setId(this.maxRowValue());

		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		EntityTransaction transaction = null;
		try {
			transaction = entityManager.getTransaction();
			transaction.begin();
			entityManager.persist(entity);
			transaction.commit();
			id = entity.getId();
		} catch (RuntimeException e) {
			logger.error("RequestDao:  Study Approval Creation failed. Transaction Rolled back.", e);
			if (transaction != null)
				transaction.rollback();
		} finally {
			entityManager.close();
		}
		return id;
	}

	/**
	 * This method is used to updateStudyApproval.
	 * @param entity The entity.
	 * @return int This returns entity id.
	 */
	@Override
	public int updateStudyApproval(StudyApproval entity) {
		hibernateTemplate.saveOrUpdate(entity);
		return entity.getId();
	}

	/**
	 * This method is used to getStudyApprovalsByID.
	 * @param id The id.
	 * @return StudyApproval This returns StudyApproval.
	 */
	@Override
	public StudyApproval getStudyApprovalsByID(int id) {
		return hibernateTemplate.get(StudyApproval.class, id);
	}

	/**
	 * This method is used to getStudyApprovalsByUID.
	 * @param uid The uid.
	 * @return this returns list of StudyApproval.
	 */
	@Override
	public List<StudyApproval> getStudyApprovalsByUID(String uid) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<StudyApproval> query = criteriaBuilder.createQuery(StudyApproval.class);
		Root<StudyApproval> root = query.from(StudyApproval.class);
		query.where(criteriaBuilder.equal(root.get("createdBy"), uid));
		query.select(root);
		return entityManager.createQuery(query).getResultList();
	}

	/**
	 * This method is used to getStudyApprovals.
	 * @return this returns list of StudyApproval.
	 */
	@Override
	public List<StudyApproval> getStudyApprovals() {
		return (List<StudyApproval>) hibernateTemplate.find("from StudyApproval");
	}

	/**
	 * Max row value.
	 *
	 * @return int This returns StudyApproval id.
	 */
	@SuppressWarnings("unchecked")
	private int maxRowValue() {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(StudyApproval.class);
		Root root = criteriaQuery.from(StudyApproval.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("id")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();
		return result.intValue() + 1;
	}

}
